/*
  --------------------- lcd.c ---------------------
*/

#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/sysinfo.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include "lcdfuncs.h"

#define WAITTIME 250

#define NET_NUM_INTERFACES 4

#define NETWEIGHTBAR (0.5)
#define NETWEIGHTNUM (0.9)

#define CPUWEIGHTBAR (0.5)
#define CPUWEIGHTNUM (0.9)

#define LOOPTIMEWEIGHT (0.95)
#define LOOPTIMESTART  16.7

/* functions */
static void initscreen(const char *name);
static void getloadpids(struct sysinfo *info);
static void getuptime(struct sysinfo *info);
static void getcpu(void);
static void getnet(float looptime);

static void blink(int on);
static void gobackground(void);
static void gotusr1(int sig);
static void gotusr2(int sig);

/* vars */
static int needsredraw = 0;
static int needsstat = 0;

int main(int argc, char *argv[])
{
  float looptime;
  int countloop = 0;
  int firsttime = 2;
  const char *name = "xxxxxxxx";
  
  struct timeval lasttv = {};
  int opt;
  struct sysinfo info;
  
  chdir("/");
  
  while ( (opt = getopt(argc, argv, "bn:")) != -1 )
    {
      switch (opt)
        {
        case 'b':
          gobackground();
          break;
          
        case 'n':
          name = optarg;
          break;
          
        default:
          break;
        }
    }
  
  signal(SIGUSR1,gotusr1);
  signal(SIGUSR2,gotusr2);
  
  initscreen(name);
  
  looptime = LOOPTIMESTART;
  
  sysinfo(&info);
  
  while (1)
    {
      struct timeval thistv;
      int changesec;
      
      gettimeofday(&thistv, NULL);
      
      if (lasttv.tv_sec != thistv.tv_sec)
        {
          changesec=1;
        }
      else
        {
          changesec=0;
        }
      
      lasttv = thistv;
      
      if (needsredraw)
        {
          needsredraw=0;
          lcd_redraw();
        }
      
      if (needsstat)
        {
          needsstat=0;
          lcd_printstats();
        }
      
#if 1
      /* every 60 seconds */
      if (changesec && !(thistv.tv_sec%60))
        {
          needsredraw=1;
        }
#endif
      
      /* every 5 seconds */
      if (changesec && !(thistv.tv_sec%5))
        {
          
          sysinfo(&info); /* update data */
          
          /* loads/pids */
          getloadpids(&info);
          
          /* uptime */
          getuptime(&info);
        }
      
      /* every second */
      if (changesec)
        {
          blink(thistv.tv_sec%2);
          
          /* update loop counter every 10 sec */
          if (firsttime)
            {
              firsttime--;
            }
          else
            {
              looptime *= LOOPTIMEWEIGHT;
              looptime += (1-LOOPTIMEWEIGHT) * countloop;
	  }
          countloop=0;
        }
      
      /* getnet */
      getnet(looptime);
      
      /* cpu meter */
      getcpu();

      lcd_flush();
      
      /* sleep */
      usleep(WAITTIME*1000);
      
      /* begin loop */
      countloop++;
      
    }
}

static void initscreen(const char *name)
{
  lcd_init(0);
  
  /* setup display                  "-123456789-123456789-123456789-123456789" */
  lcd_printf(0, 0, LCD_WIDTH, 0, 0, " --- %8s ---  nnnnnkB[        ] LAN", name);
  lcd_printf(1, 0, LCD_WIDTH, 0, 0, "UPTIME xxxxxxxxx   nnnnnkB[        ] LEX");
  lcd_printf(2, 0, LCD_WIDTH, 0, 0, "PIDs xxx LOAD xxxx nnnnnkB[        ] WAN");
  lcd_printf(3, 0, LCD_WIDTH, 0, 0, "CPU nnn%%[        ] nnnnnkB[        ]WIFI");
  
  return;
}

static void getloadpids(struct sysinfo *info)
{
  float l0;
  
  l0 = ((float)info->loads[0])/(float)(1<<(int)SI_LOAD_SHIFT);
  
  if (l0 >= 100.0)
    {
      lcd_printf(2, 14, LCD_WIDTH, 4, 0, "%4.0f", l0);
    }
  else if (l0 >= 10.0)
    {
      lcd_printf(2, 14, LCD_WIDTH, 4, 0, "%4.1f", l0);
    }
  else
    {
      lcd_printf(2, 14, LCD_WIDTH, 4, 0, "%4.2f", l0);
    }
  
  /* total pids */
  lcd_printf(2, 5, LCD_WIDTH, 4, 0, "%u", info->procs);
  return;
}

static void getuptime(struct sysinfo *info)
{
  /* convert uptime */
  if (info->uptime >= 86400)
    {
      lcd_printf(1, 7, LCD_WIDTH, 9, 0, "%lud %luh",
                 info->uptime/86400,
                 (info->uptime/3600)-(24*(info->uptime/86400)));
    }
  else
    {
      lcd_printf(1, 7, LCD_WIDTH, 9, 0, "%luh %lum",
                 (info->uptime/3600),
                 info->uptime/60-60*(info->uptime/3600));
    }
  
  return;
}

static void getcpu(void)
{
  FILE *fptr;
  
  fptr=fopen("/proc/stat", "r");
  if (fptr == NULL)
    {
      lcd_printf(3, 0, LCD_WIDTH, 0, 0, "/proc/stat: %s", strerror(errno));
    }
  else
    {
      char buffer[100];
      
      while (fgets(buffer, 100, fptr) != NULL)
        {
          long long cur[7];
          
          /* user nice system idle iowait irq softirq */
          if (sscanf(buffer,"cpu  %Lu %Lu %Lu %Lu %Lu %Lu %Lu",
                     &cur[0],&cur[1],&cur[2],&cur[3],&cur[4],&cur[5],&cur[6]) == 7)
            {
              long long used, idle;
              static long long last_used = 0;
              static long long last_idle = 0;
              static float cpuvalbar = 0.0;
              static float cpuvalnum = 0.0;
              
              used = (cur[0] + cur[1] + cur[2] + cur[4] + cur[5] + cur[6]);
              idle = cur[3];
              
              cpuvalbar *= CPUWEIGHTBAR;
              cpuvalbar += (1-CPUWEIGHTBAR)*(((float)used-last_used)/(((float)used-last_used)+((float)idle-last_idle)));
              
              cpuvalnum *= CPUWEIGHTNUM;
              cpuvalnum += (1-CPUWEIGHTNUM)*(((float)used-last_used)/(((float)used-last_used)+((float)idle-last_idle)));
              
              last_used = used;
              last_idle = idle;
              
              lcd_putbar(3, 8, 10, (int)(50*cpuvalbar), 0);
              
              lcd_printf(3, 4, LCD_WIDTH, 3, 0, "%3.0f", 100*cpuvalnum);
              
              break;
            }
        }
      
      fclose(fptr);
    }
  
  return;
}

static void getnet(float looptime)
{
  FILE *fptr;
  
  fptr=fopen("/proc/net/dev", "r");
  if (fptr == NULL)
    {
      lcd_printf(0, 20, LCD_WIDTH, 0, 0, "/proc/net/dev: %s", strerror(errno));
    }
  else
    {
      char buffer[200];
      
      while (fgets(buffer, 200, fptr))
        {
          long long in, out;
          unsigned int dev;
          char *buffer2;
          
          buffer2 = strchr(buffer,':');
          if (!buffer2)
            {
              continue;
            }
          
          if (sscanf(buffer2+1,
                     "%Lu %*u %*u %*u %*u %*u %*u %*u "
                     "%Lu %*u %*u %*u %*u %*u %*u %*u",
                     &in,&out) == 2)
            {
              static long long last_in[NET_NUM_INTERFACES] = {};
              static long long last_out[NET_NUM_INTERFACES] = {};
              static float netvalbar[NET_NUM_INTERFACES] = {};
              static float netvalnum[NET_NUM_INTERFACES] = {};
              
              if (!strncmp(buffer,"  eth0:",7))
                {
                  dev = 0;
                }
              else if (!strncmp(buffer,"eth2.240:",9))
                {
                  dev = 1;
                }
              else if (!strncmp(buffer,"  eth1:",7))
                {
                  dev = 2;
                }
              else if (!strncmp(buffer,"eth2.230:",9))
                {
                  dev = 3;
                }
              else
                {
                  continue;
                }
              
              if (!last_in[dev] && !last_out[dev])
                {
                  /* first run */
                  last_in[dev]  = in;
                  last_out[dev] = out;
                  continue;
                }
              
              netvalbar[dev] *= NETWEIGHTBAR;
              netvalbar[dev] += (1-NETWEIGHTBAR)*( ((float)(in+out-last_in[dev]-last_out[dev]))*looptime );
              
              netvalnum[dev] *= NETWEIGHTNUM;
              netvalnum[dev] += (1-NETWEIGHTNUM)*( ((float)(in+out-last_in[dev]-last_out[dev]))*looptime );
              
              last_in[dev]  = in;
              last_out[dev] = out;
              
              lcd_putbar(dev, 26, 10, netvalbar[dev]/1024/12.0, 0); /* 1bar = 12K, 0..600K */
              
              lcd_printf(dev, 19, 5, 5, 0, "%5.0f", netvalnum[dev]/1024);
            }
        }
      
      fclose(fptr);
    }
  
  return;
}


static void blink(int on)
{
  const char *ss;
  
  ss = on ? "\xFF\xFF\xFF" : "===";
  
  lcd_printf(0, 1,  LCD_WIDTH, 0, 0, ss);
  lcd_printf(0, 14, LCD_WIDTH, 0, 0, ss);
  
  return;
}

/* misc stuff */

static void gobackground(void)
{
  int s;
  /* parent forks */
  s = fork();
  if (s < 0)
    {
      lcd_printf(0, 0, LCD_WIDTH, 0, 1, "Unable to Fork1: %s", strerror(errno));
    }
  else if (s > 0)
    {
      exit(0);
    }
  
  /* stdin   stdout    stderr */
  close(0); close(1); close(2);
  
  s = setsid();
  if (s < 0)
    {
      lcd_printf(0, 0, LCD_WIDTH, 0, 1, "Couldn't setsid: %s", strerror(errno));
    }
  
  /* parent forks */
  s = fork();
  if (s < 0)
    {
      lcd_printf(0, 0, LCD_WIDTH, 0, 1, "Unable to Fork1: %s", strerror(errno));
    }
  else if (s > 0)
    {
      /* parent exits */
      exit(0);
    }
  
  s = open("/dev/null", O_RDWR); /* stdin */
  dup(s);                        /* stdout */
  dup(s);                        /* stderr */
  
  return;
}

static void gotusr1 (int sig)
{
  needsredraw=1;
  signal(SIGUSR1,gotusr1);
  
  return;
}

static void gotusr2 (int sig)
{
  needsstat=1;
  signal(SIGUSR2,gotusr2);
  
  return;
}

/* end of file */

