/*
  ------------------- lcdfuncs.c -------------------
  low level functions for controlling the lcd screen
*/

#include "lcdfuncs.h"
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/io.h>
#include <sys/time.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#ifdef SOEKRIS
#include <linux/net4801gpio.h>
#else
#include <linux/ppdev.h>
#endif

typedef struct { long writes, requests, seeks, nexts, flushs; } lcd_stats_t;

#ifdef SOEKRIS

#define GPIO_DEV "/dev/gpio0"

#define REG_CONTROL_RS   0x01
#define REG_CONTROL_E2   0x02
#define REG_CONTROL_E1   0x04
#define REG_CONTROL_RW   0x08

#else

/* pins used for lcd screen

  || port   M4024
  -------   -----
   D0-D7  - DB0-DB7
   ^C0    - E1
   ^C1    - E2
   ^C2    - RS
   
   control = (c)xxxx3210 (3,1,0 inverted)
*/

#define GPIO_DEV "/dev/parport0"

#define REG_CONTROL_E1   0x01
#define REG_CONTROL_E2   0x02
#define REG_CONTROL_RS   0x04
#define REG_CONTROL_RW   0

#endif

#define LCD_4BIT_MODE 1

typedef enum
{
  LCD_RS_INST         = 0,
  LCD_RS_DATA         = REG_CONTROL_RS,
} rs_t;

typedef enum 
{
  LCD_TOP             = REG_CONTROL_E1,
  LCD_BOTTOM          = REG_CONTROL_E2,
  LCD_TOP_AND_BOTTOM  = REG_CONTROL_E1 | REG_CONTROL_E2
} where_t;


/* local function declarations */
static void lcd_init_screen(int clear);
static void lcd_write(rs_t rs, where_t where, unsigned char cmd);
static void lcd_gotopos(char row, char col);
static short lcd_barcharnum(short num, char last);
static void lcd_initchars(void);
static void lcd_set_control(unsigned char data);
static void lcd_set_data(unsigned char data);
static void lcd_setcustchar(char charnum, where_t where,
			    char r1, char r2, char r3, char r4,
			    char r5, char r6, char r7, char r8);
static void spin(int usec);

/* global variables */
static char set_custom_mode;
static unsigned char cbuff[LCD_HEIGHT][LCD_WIDTH];
static unsigned char cbufflast[LCD_HEIGHT][LCD_WIDTH];
static lcd_stats_t lcd_stats;

static int gpio_dev_fd = -1;

/* macros */
#define BITS_TO_CHAR(d7,d6,d5,d4,d3,d2,d1,d0) \
	 (((d7) << 7) | \
	  ((d6) << 6) | \
	  ((d5) << 5) | \
	  ((d4) << 4) | \
	  ((d3) << 3) | \
	  ((d2) << 2) | \
	  ((d1) << 1) | \
	  ((d0)     ) )

#define BITS5_TO_CHAR(d4,d3,d2,d1,d0) \
	 (((d4) << 4) | \
	  ((d3) << 3) | \
	  ((d2) << 2) | \
	  ((d1) << 1) | \
	  ((d0)     ) )


/* API functions */

void lcd_init (char custom_mode)
{
#ifdef SOEKRIS
  int arg;
  
  if (gpio_dev_fd < 0)
    {
      gpio_dev_fd = open(GPIO_DEV, O_RDWR);
      if (gpio_dev_fd < 0)
        {
          perror("Cannot open " GPIO_DEV);
          exit(1);
        }
    }
  
  if (ioctl(gpio_dev_fd, GPIORDNUMPINS, &arg) < 0)
    {
      perror("cant get numpins");
      exit(1);
    }
  if (arg < 8)
    {
      fprintf(stderr,"not enough pins, only %d\n", arg);
      exit(1);
    }
  
  /* set all pins to output */
  arg = 0xFF;
  if (ioctl(gpio_dev_fd, GPIOWRDIRECTION, &arg) < 0)
    {
      perror("cant set direction");
      exit(1);
    }

#else
  if (gpio_dev_fd < 0)
    {
      gpio_dev_fd = open(GPIO_DEV, O_RDWR);
      if (gpio_dev_fd < 0)
        {
          perror("Cannot open " GPIO_DEV);
          exit(1);
        }
      
      if (ioctl(gpio_dev_fd, PPCLAIM, NULL) < 0)
        {
          perror("cannot claim pport");
          exit(1);
        }
    }

#endif
  
  if (nice(0) < 0)
    {
      perror("Cant nice to -10");
    }
  
  /* initialize registers */
  lcd_set_data(0);
  lcd_set_control(0);
  spin(2000);
  
  /* setup lcd screen */
  
  
  set_custom_mode = custom_mode;
  
  lcd_init_screen(1);
  
  if (set_custom_mode == 0)
    {
      /* setup custom characters */
      lcd_initchars();
    }
  
  bzero (&lcd_stats,sizeof(lcd_stats_t));
  
  /* clear display */
  lcd_clear();
  
  /* lcd setup complete */
  
}

void lcd_uninit(void)
{
#ifdef SOEKRIS
#else
  if (ioctl(gpio_dev_fd, PPRELEASE, NULL) < 0)
    {
      perror("cannot release pport");
      exit(1);
    }
#endif
  close(gpio_dev_fd);
  gpio_dev_fd = -1;
  return;
}

/* clears and returns home */
void lcd_clear(void)
{
  lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,0,0, 0,0,0,1));
  spin(2000);
  
  memset(&cbuff[0][0],    ' ',LCD_WIDTH*LCD_HEIGHT);
  memset(&cbufflast[0][0],' ',LCD_WIDTH*LCD_HEIGHT);
}

/* redraws and returns home */
void lcd_redraw(void) {
  
  memset(&cbufflast[0][0],'\x80',LCD_WIDTH*LCD_HEIGHT);
  
  lcd_init_screen(0);
  
  lcd_flush();

  if (set_custom_mode == 0)
    {
      /* setup custom characters */
      lcd_initchars();
    }
  
}

/* turns the display on/off */
/* t: 0=off, 1=on */
void lcd_onoff(char t) {
  
  /* display on/off control          */
  /* cursor blink on/off ----------\ */
  /* cursor on/off --------------\ | */
  /* display on/off -----------\ | | */
  lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,0,0, 1,t,0,0));
  spin(40);
  
}

/* writes a string to any part of the display */
int __attribute__ ((format(printf, 6, 7)))
     lcd_printf(int row, int col, int max_length, int min_length, int flush,
                const char *format, ...)
{
  int i;
  char tempstr[LCD_WIDTH+1];
  va_list args;
  
  if ((row < 0) || (row > 3) || (col < 0) || (col > LCD_WIDTH-1) ||
      ((col + min_length) > LCD_WIDTH) ||
      (min_length > max_length))
    {
      return 0;
    }
  
  va_start(args, format);
  vsnprintf(tempstr,LCD_WIDTH+1,format,args);
  tempstr[LCD_WIDTH] = '\0';
  va_end(args);
  
  for (i=0; i<min_length; i++)
    {
      cbuff[row][col+i] = ' ';
      lcd_stats.requests++;
    }
  
  for (i=0; ((col+i)<LCD_WIDTH) && (tempstr[i] != '\0') && i<max_length; i++)
    {
      cbuff[row][col+i] = (unsigned char)tempstr[i];
      if (i >= min_length)
        {
          lcd_stats.requests++;
        }
    }
  
  if (flush)
    {
      lcd_flush();
    }
  return i;
}

/* displays a bar on the lcd in row:col-col+wide with val */
/* val should be normalized to 1..wide*5         */
void lcd_putbar (int row, int col, int wide, short val, int flush) {
  int i;
  val++; 
  
  if (set_custom_mode != 0) return;

  if (row < 0 || row > 3 || col < 0 || col > 39) return;
  
  if (val > 5*wide) val = 5*wide;
  if (val < 1) val = 1;
  
  for (i=0; ((col+i)<39 && i<wide-1); i++) {
    cbuff[row][col+i] = lcd_barcharnum(val-(5*i),0);
    lcd_stats.requests++;
  }
  cbuff[row][col+i] = lcd_barcharnum(val-(5*(wide-1)),1);
  lcd_stats.requests++;
  
  if (flush)
    {
      lcd_flush();
    }
}

void lcd_locategraphic (int col)
{
  int i,j;
  
  /* set */
  for (i=0; i<2; i++)
    for (j=0; j<4; j++)
      cbuff[i][j+col] = cbuff[i+2][j+col] = j + (i*4);
  
  lcd_flush();
  
}

void lcd_updategraphic (long *data)
{
  int i;
  
  /* setup custom chars, top */
  for (i=0; i<4; i++)
    {
      lcd_setcustchar(i,LCD_TOP,
		      (data[0] >> (6*(3-i))) & 0x1F,
		      (data[1] >> (6*(3-i))) & 0x1F,
		      (data[2] >> (6*(3-i))) & 0x1F,
		      (data[3] >> (6*(3-i))) & 0x1F,
		      (data[4] >> (6*(3-i))) & 0x1F,
		      (data[5] >> (6*(3-i))) & 0x1F,
		      (data[6] >> (6*(3-i))) & 0x1F,
		      (data[7] >> (6*(3-i))) & 0x1F);
    }
  
  for (i=0; i<4; i++)
    {
      lcd_setcustchar(i+4,LCD_TOP,
		      (data[10+0] >> (6*(3-i))) & 0x1F,
		      (data[10+1] >> (6*(3-i))) & 0x1F,
		      (data[10+2] >> (6*(3-i))) & 0x1F,
		      (data[10+3] >> (6*(3-i))) & 0x1F,
		      (data[10+4] >> (6*(3-i))) & 0x1F,
		      (data[10+5] >> (6*(3-i))) & 0x1F,
		      (data[10+6] >> (6*(3-i))) & 0x1F,
		      (data[10+7] >> (6*(3-i))) & 0x1F);
    }
  
  /* setup custom chars, bottom */
  for (i=0; i<4; i++)
    {
      lcd_setcustchar(i,LCD_BOTTOM,
		      (data[20+0] >> (6*(3-i))) & 0x1F,
		      (data[20+1] >> (6*(3-i))) & 0x1F,
		      (data[20+2] >> (6*(3-i))) & 0x1F,
		      (data[20+3] >> (6*(3-i))) & 0x1F,
		      (data[20+4] >> (6*(3-i))) & 0x1F,
		      (data[20+5] >> (6*(3-i))) & 0x1F,
		      (data[20+6] >> (6*(3-i))) & 0x1F,
		      (data[20+7] >> (6*(3-i))) & 0x1F);
    }
  
  for (i=0; i<4; i++)
    {
      lcd_setcustchar(i+4,LCD_BOTTOM,
		      (data[30+0] >> (6*(3-i))) & 0x1F,
		      (data[30+1] >> (6*(3-i))) & 0x1F,
		      (data[30+2] >> (6*(3-i))) & 0x1F,
		      (data[30+3] >> (6*(3-i))) & 0x1F,
		      (data[30+4] >> (6*(3-i))) & 0x1F,
		      (data[30+5] >> (6*(3-i))) & 0x1F,
		      (data[30+6] >> (6*(3-i))) & 0x1F,
		      (data[30+7] >> (6*(3-i))) & 0x1F);
    }
  
  
}

void lcd_printstats (void) {
  unsigned char oldbuff[LCD_HEIGHT][LCD_WIDTH];
  
  memcpy(&oldbuff[0][0],&cbuff[0][0],LCD_HEIGHT*LCD_WIDTH);
  
  lcd_clear();
  
  lcd_printf(0, 0, LCD_WIDTH, 0, 0, "LCD API Statistics:");
  
  lcd_printf(1, 0, LCD_WIDTH, 0, 0, "%5.2e write, %5.2e req,  %2.1f%% hit",
             (float)lcd_stats.writes,
             (float)lcd_stats.requests,
             100.0-(100.0*(float)lcd_stats.writes/(float)lcd_stats.requests));
  
  lcd_printf(2, 0, LCD_WIDTH, 0, 0, "%5.2e next,  %5.2e seek, %2.1f%% hit",
             (float)lcd_stats.nexts,
             (float)lcd_stats.seeks,
             100.0*(float)lcd_stats.nexts/(float)lcd_stats.writes);
  
  lcd_printf(3, 0, LCD_WIDTH, 0, 0, "%5.2e flush, %2.1f res/f, %2.1f wrts/f",
             (float)lcd_stats.flushs,
             (float)lcd_stats.requests/(float)lcd_stats.flushs,
             (float)lcd_stats.writes/(float)lcd_stats.flushs);
  
  lcd_flush();
  sleep(15);
  lcd_clear();
  
  memcpy(&cbuff[0][0],&oldbuff[0][0],LCD_HEIGHT*LCD_WIDTH);
  lcd_flush();
  
}

/* internal functions */

static void lcd_init_screen(int clear)
{
  
#if LCD_4BIT_MODE
  /* set to 4 bit mode */
  lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,1,0, 1,0,0,0));
#else
  /* set to 8 bit mode */
  lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,1,1, 1,0,0,0));
#endif
  spin(40);
  
  /* set entry mode                                                        */
  /* shift entire display -----------------------------------------------\ */
  /* shift cursor -----------------------------------------------------\ | */
  lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,0,0, 0,1,1,0));
  spin(2000);
  
  /* display on/off control                                                */
  /* cursor blink on/off ------------------------------------------------\ */
  /* cursor on/off ----------------------------------------------------\ | */
  /* display on/off -------------------------------------------------\ | | */
  lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,0,0, 1,1,0,0));
  spin(40);
  
  if (clear)
    {
      /* clear and return home */
      lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,0,0, 0,0,0,1));
      spin(2000);
    }
  else
    {
      /* return home */
      lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,0,0, 0,0,1,0));
      spin(2000);
    }
  
  /* return home */
  lcd_write(LCD_RS_INST, LCD_TOP_AND_BOTTOM, BITS_TO_CHAR(0,0,0,0, 0,0,1,0));
  spin(2000);
  
  return;
}

#ifdef SOEKRIS

static void lcd_set_reg(unsigned int data, unsigned int mask)
{
  static int cur_reg = 0;
  
  cur_reg &= ~mask;
  cur_reg |= data;
  
#if 0
  printf("cur=0x%.2X, data=0x%.2X, mask=0x%.2X: %2s %2s %2s %2s data=0x%.1X\n",
         cur_reg,
         data, mask,
         cur_reg & REG_CONTROL_RW ? "rw" : "",
         cur_reg & REG_CONTROL_RS ? "rs" : "",
         cur_reg & REG_CONTROL_E1 ? "e1" : "",
         cur_reg & REG_CONTROL_E2 ? "e2" : "",
         (cur_reg & 0xF0) >> 4);
#endif
  
  if (ioctl(gpio_dev_fd, GPIOWRDATA, &cur_reg) < 0)
    {
      perror("cant set data");
      exit(1);
    }
}

static void lcd_set_control(unsigned char data)
{
  lcd_set_reg(data, (REG_CONTROL_RS | REG_CONTROL_RW | REG_CONTROL_E1 | REG_CONTROL_E2));
}
static void lcd_set_data(unsigned char data)
{
  lcd_set_reg(data, 0xFF & ~(REG_CONTROL_RS | REG_CONTROL_RW | REG_CONTROL_E1 | REG_CONTROL_E2));
}

#else

static void lcd_set_control(unsigned char data)
{
  /* E1 and E2 are inverted */
  data ^= REG_CONTROL_E1 | REG_CONTROL_E2;
  data |= 0x08;
  data &= 0x0F;
  
  if (ioctl(gpio_dev_fd, PPWCONTROL, &data) < 0)
    {
      perror("cant set control reg");
      exit(1);
    }
}
static void lcd_set_data(unsigned char data)
{
  if (ioctl(gpio_dev_fd, PPWDATA, &data) < 0)
    {
      perror("cant set control reg");
      exit(1);
    }
}

#endif


static void lcd_write(rs_t rs, where_t where, unsigned char cmd)
{
#if LCD_4BIT_MODE
  /* upper 4 bits */
  lcd_set_control(rs);
  spin(1); /* spin_ns(140); */
  
  lcd_set_control(rs | where);
  lcd_set_data(cmd & 0xF0);
  spin(1); /* spin_ns(230); */
  
  lcd_set_control(rs);
  spin(1); /* spin_ns(130); */

  /* lower 4 bits */
  lcd_set_control(rs);
  spin(1); /* spin_ns(140); */
  
  lcd_set_control(rs | where);
  lcd_set_data((cmd << 4) & 0xF0);
  spin(1); /* spin_ns(230); */
  
  lcd_set_control(rs);
  spin(1); /* spin_ns(130); */
#else
  lcd_set_control(rs);
  spin(1); /* spin_ns(140); */
  
  lcd_set_control(rs | where);
  lcd_set_data(cmd);
  spin(1); /* spin_ns(230); */
  
  lcd_set_control(rs);
  spin(1); /* spin_ns(130); */
#endif
  return;
}

/* goto position on lcd screen 0..3 and 0..39 */
static void lcd_gotopos(char row, char col)
{
  if (col > 39)
    {
      return;
    }
  
  if      (row == 0) lcd_write(LCD_RS_INST, LCD_TOP,    0x80+col);
  else if (row == 1) lcd_write(LCD_RS_INST, LCD_TOP,    0xC0+col);
  else if (row == 2) lcd_write(LCD_RS_INST, LCD_BOTTOM, 0x80+col);
  else if (row == 3) lcd_write(LCD_RS_INST, LCD_BOTTOM, 0xC0+col);
  
  spin(40);
}

static short lcd_barcharnum(short num, char last)
{
  /* num is 0..5 */
  if (!last) {
    if (num < 1) return 7;
    else if (num > 4) return 4;
    else return (num-1);
  }
  else {
    if (num < 1) return 5;
    else if (num < 2) return (num+5);
    else if (num > 4) return 4;
    else return (num-1);
  }
}

static void lcd_setcustchar(char charnum, where_t where,
			    char r1, char r2, char r3, char r4,
			    char r5, char r6, char r7, char r8)
{
  char loc2, loc1, loc0;
  loc2 = (charnum >> 2) & 0x01;
  loc1 = (charnum >> 1) & 0x01;
  loc0 = (charnum >> 0) & 0x01;
  
  /* CG Address                                                               */
  /* line of character 0-8 -----------------------------------------\-\-\     */
  /* character 0-7 --------------------------------|----|----|      | | |     */
  /* CG Data                                                                  */
  /* data for 5 columns                                                       */
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  0,0,0));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r1 & 0x1F);
  spin(40);
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  0,0,1));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r2 & 0x1F);
  spin(40);
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  0,1,0));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r3 & 0x1F);
  spin(40);
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  0,1,1));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r4 & 0x1F);
  spin(40);
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  1,0,0));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r5 & 0x1F);
  spin(40);
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  1,0,1));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r6 & 0x1F);
  spin(40);
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  1,1,0));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r7 & 0x1F);
  spin(40);
  
  lcd_write(LCD_RS_INST, where, BITS_TO_CHAR(0,1,  loc2,loc1,loc0,  1,1,1));
  spin(40);
  lcd_write(LCD_RS_DATA, where, r8 & 0x1F);
  spin(40);
}


static void lcd_initchars(void)
{
  /* character 0: 1 line */
  lcd_setcustchar(0,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,0,0,0,0),
		  BITS5_TO_CHAR(1,0,0,0,0),
		  BITS5_TO_CHAR(1,0,0,0,0),
		  BITS5_TO_CHAR(1,0,0,0,0),
		  BITS5_TO_CHAR(1,0,0,0,0),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
  
  /* character 1: 2 lines */
  lcd_setcustchar(1,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,0,0,0),
		  BITS5_TO_CHAR(1,1,0,0,0),
		  BITS5_TO_CHAR(1,1,0,0,0),
		  BITS5_TO_CHAR(1,1,0,0,0),
		  BITS5_TO_CHAR(1,1,0,0,0),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
  
  /* character 2: 3 lines */
  lcd_setcustchar(2,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,0,0),
		  BITS5_TO_CHAR(1,1,1,0,0),
		  BITS5_TO_CHAR(1,1,1,0,0),
		  BITS5_TO_CHAR(1,1,1,0,0),
		  BITS5_TO_CHAR(1,1,1,0,0),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
  
  /* character 3: 4 lines */
  lcd_setcustchar(3,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,1,0),
		  BITS5_TO_CHAR(1,1,1,1,0),
		  BITS5_TO_CHAR(1,1,1,1,0),
		  BITS5_TO_CHAR(1,1,1,1,0),
		  BITS5_TO_CHAR(1,1,1,1,0),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
  
  /* character 4: 5 lines */
  lcd_setcustchar(4,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
  
  /* character 5: 1 end line */
  lcd_setcustchar(5,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,1),
		  BITS5_TO_CHAR(0,0,0,0,1),
		  BITS5_TO_CHAR(0,0,0,0,1),
		  BITS5_TO_CHAR(0,0,0,0,1),
		  BITS5_TO_CHAR(0,0,0,0,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
  
  /* character 6: 1 + 1 end */
  lcd_setcustchar(6,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(1,0,0,0,1),
		  BITS5_TO_CHAR(1,0,0,0,1),
		  BITS5_TO_CHAR(1,0,0,0,1),
		  BITS5_TO_CHAR(1,0,0,0,1),
		  BITS5_TO_CHAR(1,0,0,0,1),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
  
  /* character 7: space */
  lcd_setcustchar(7,LCD_TOP_AND_BOTTOM,
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0),
		  BITS5_TO_CHAR(0,0,0,0,0),
		  BITS5_TO_CHAR(0,0,0,0,0),
		  BITS5_TO_CHAR(0,0,0,0,0),
		  BITS5_TO_CHAR(0,0,0,0,0),
		  BITS5_TO_CHAR(1,1,1,1,1),
		  BITS5_TO_CHAR(0,0,0,0,0));
}

void lcd_flush(void)
{
  int i,j,lastwritten;
  
  lcd_stats.flushs++;
  
  for (i=0; i<LCD_HEIGHT; i++) {
    lastwritten=-2;
    for (j=0; j<LCD_WIDTH; j++)
      if (cbuff[i][j] != cbufflast[i][j]) {
	if (lastwritten+1 != (i>>1)*LCD_WIDTH+j) {
	  lcd_gotopos(i,j);
	  lcd_stats.seeks++;
	}
	else
	  lcd_stats.nexts++;
	lcd_write(LCD_RS_DATA, (i<2) ? LCD_TOP : LCD_BOTTOM, cbuff[i][j]);
        spin(40);
	cbufflast[i][j] = cbuff[i][j];
	lastwritten = (i>>1)*LCD_WIDTH+j;
	lcd_stats.writes++;
      }
  }
  fflush(stdout);
}

static void spin(int usec)
{
  struct timeval now_tv;
  struct timeval end_tv;
  long interval;
  
  if (gettimeofday(&now_tv, NULL) < 0)
    {
      return;
    }
  
  end_tv = now_tv;
  
  end_tv.tv_usec += usec;
  while(end_tv.tv_usec >= 1000000)
    {
      end_tv.tv_sec++;
      end_tv.tv_usec -= 1000000;
    }
  
  while (1)
    {
      interval = (end_tv.tv_sec - now_tv.tv_sec)*1000000
        + (end_tv.tv_usec - now_tv.tv_usec);
      
      if (interval > 10000)
        {
          usleep(interval);
        }
      
      if (gettimeofday(&now_tv, NULL) < 0)
        {
          return;
        }
      
      if (now_tv.tv_sec > end_tv.tv_sec)
        {
          break;
        }
      else if ((now_tv.tv_sec == end_tv.tv_sec) &&
               (now_tv.tv_usec >= end_tv.tv_usec))
        {
          break;
        }
    }
  
  return;
}


/* end of file */
